; HW_window.asm - hello world from a window, probably the second simplest
; Win32 program
;

%define _WINMESSAGES_
%include "Gaz\Win32\Include\Windows.inc"

[BITS 32]
[section .text]

procglobal WinMain, hInstance, hPrevInstance, lpszCmdLine, nCmdShow
	;
	; Locals - window handle, window class structure and message structure
	;
	ddlocal		_hwnd
	struclocal	_wndclass, WNDCLASSEX, _msg, MSG
	endlocals
	;
	WinMainPrologue
	;
	; To display a window, we need to first declare a Window Class, which we
	; do by populating a WNDCLASSEX data structure (as defined in the locals
	; above). First, we point esi and edi to the two structures so we can
	; populate the WNDCLASSEX structure and easily access the MSG one later.
	;
	; Note the dot prefix as they're both locals
	;
	mov	esi, ._wndclass
	mov	edi, ._msg
	;
	; First thing to set is the structure size. There are several structures
	; in Win32 which are like this and calls using them will fail if you
	; forget to set the size. Because it would be a pain to have to remember
	; or work out the offsets of each member of the data structure, we can use
	; NASM's power to let it do the hard work, and refer to the offset via
	; <STRUCTURE_NAME>.<MEMBER_NAME>, which will work for any structure.
	; Handily, every structure defined in NASM automatically has its size
	; defined as <STRUCTURE_NAME>_size:
	;
	mov	[esi + WNDCLASSEX.cbSize], dword WNDCLASSEX_size
	;
	; The style determines when the window is to be repainted, the two
	; constants here meaning when it is resized vertically or horizontally
	;
	mov	[esi + WNDCLASSEX.style], dword CS_HREDRAW | CS_VREDRAW
	;
	; lpfnWndProc holds the address of the Window Procedure, which is a
	; user-defined function which processes any messages sent to a window
	;
	mov	[esi + WNDCLASSEX.lpfnWndProc], dword _WndProc
	;
	; The next two fields are reserved and set to 0
	;
	mov	[esi + WNDCLASSEX.cbClsExtra], dword 0
	mov	[esi + WNDCLASSEX.cbWndExtra], dword 0
	;
	; hInstance is the instance handle of the program, and is passed to
	; WinMain upon program start (amongst other things). Note that because
	; it's a parameter passed to our proc we prefix it with a dot
	;
	mov	eax, .hInstance
	mov	[esi + WNDCLASSEX.hInstance], eax
	;
	; hIcon holds an Icon Handle, which is a handle to the icon that will
	; be used on the taskbar. We call the API function LoadIcon to return
	; a handle to a built-in (to Win32) icon. See the Win32 help file for
	; more information on LoadIcon and details on more of the built-in
	; icons. As with all Win32 functions, the return value is stored in
	; eax, in this case a handle to the icon, and we place this in the
	; data structure
	;
	sc LoadIcon, NULL, IDI_APPLICATION
	mov	[esi + WNDCLASSEX.hIcon], eax
	;
	; Similarly to hIcon, hCursor holds the cursor icon, and is set in
	; exactly the same way, except the API call is different
	;
	sc LoadCursor, NULL, IDC_ARROW
	mov	[esi + WNDCLASSEX.hCursor], eax
	;
	; hbrBackground holds a handle to a brush that will be used to draw
	; the window's background in. A brush is just a block of pixels, much
	; like a brush in a paint package, and like icons and cursors, Win32
	; has a number of built-in ones
	;
	sc GetStockObject, WHITE_BRUSH
	mov	[esi + WNDCLASSEX.hbrBackground], eax
	;
	; lpszMenuName is used to specify a menu for the window, but we set
	; it to NULL as we don't have one
	;
	mov	[esi + WNDCLASSEX.lpszMenuName], dword NULL
	;
	; lpszClassName holds the name of the window class so we can refer
	; to it later. It doesn't actually hold the name, but a pointer to, ie
	; in our terms the address of, a string. The use of pointers to things
	; (ie the address of a thing) is extremely common in Win32
	;
	TEXTlocal szClassName, 'MyClass',0
	mov	[esi + WNDCLASSEX.lpszClassName], dword .szClassName
	;
	; Having created the window class, we tell Windows that it can be used
	; by calling the RegisterClassEx API function, with the address of
	; the window class data structure. I've passed esi as it already holds
	; the address of the structure, though I could have used ._wndclass
	;
	sc RegisterClassEx, esi
	;
	; We can check that the call succeeded by checking the return value (in
	; eax) as it will be 0 if the call failed.
	;
	cmp	eax, 0
	je	near _WinMain_Fail
	;
	; The call succeeded, so we call CreateWindowEx to create the window,
	; passing it a variety of parameters, notably the window class name,
	; the caption and the instance handle (.hInstance). Note again the use
	; of pointers to strings and the dot prefix on .hInstance
	;
	TEXTlocal szWndCaption, 'Hello, world!',0
	sc CreateWindowEx, 0, .szClassName, .szWndCaption, WS_OVERLAPPEDWINDOW | WS_VISIBLE, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, .hInstance, NULL
	;
	; CreateWindowEx returns a handle to the window, so that we can refer
	; to it later on, so we store it in a local variable
	;
	mov	._hwnd, eax
	;
	; After creation, we need to show the window by calling ShowWindow.
	; nCmdShow is passed to WinMain upon program startup, and it holds
	; the start-up state of the window (minimised, maximised, normal, etc).
	;
	sc ShowWindow, ._hwnd, .nCmdShow
	;
	; UpdateWindow forces a window (referred to by a window handle) to
	; be redrawn so we call it here to ensure the window is visible
	;
	sc UpdateWindow, ._hwnd
_WinMain_Loop:
	;
	; Having created and displayed the window, we go into what is known
	; as the 'message loop' where we retrieve and process messages sent
	; to the window. This falls into several stages, the first of which
	; is retrieving a message from the message queue:
	;
	sc GetMessage, ._msg, NULL, 0, 0
	;
	; GetMessage returns TRUE if there is any message other than a
	; WM_QUIT message, which tells the window to close. If we get
	; that message we jump to _WinMain_Loop_End and finish the program
	;
	cmp	eax, TRUE
	jne	_WinMain_Loop_End
	;
	; We have something other than a WM_QUIT message so we call
	; TranslateMessage which performs keyboard translation (which we
	; don't need to worry about here)
	;
	sc TranslateMessage, ._msg
	;
	; And then we call DispatchMessage which tells Windows to call
	; the Window Procedure for this window
	;
	sc DispatchMessage, ._msg
	;
	; And continue looping until we get a WM_QUIT message
	;
	jmp	_WinMain_Loop
_WinMain_Loop_End:
	;
	; Exit the program via WinMainEpilogue, setting eax to the
	; return code of the WM_QUIT message
	;
	mov	eax, [edi + MSG.wParam]
	jmp	_WinMain_End
_WinMain_Fail:
	;
	; Failed to register the window class so display an error box
	; and exit
	;
	TEXTlocal szErrorMsg, 'Failed to register window class!',0
	sc MessageBox, NULL, .szErrorMsg, .szWndCaption, MB_ICONERROR
_WinMain_End:
	;
	; Finally, we call the WinMainEpilogue macro which tidies up
	; registers and exits the program
	;
	WinMainEpilogue
endproc
;
;-----------------------------------------------------------------------
;
; Window procedure - where we process any messages sent to a window
;
; Like WinMain, a window procedure takes four parameters:
;
;   1. hwnd - handle of window being processed
;   2. message - the message to be processed
;   3. wParam - this and lParam are extra message parameters
;   4. lParam
;
proc _WndProc, hwnd, message, wParam, lParam
	;
	; local variables to hold a device context handle and other structures
	;
	ddlocal		_hdc
	struclocal	_rect, RECT, _ps, PAINTSTRUCT
	endlocals
	;
	; Any procedure called by Windows is known as a Callback, and
	; CallbackPrologue preserves the registeres required by Windows.
	;
	; IT MUST COME AFTER endlocals
	;
	CallbackPrologue
	;
	; Work out which message we're processing and act accordingly
	;
	switch .message
		case WM_PAINT
			;
			; WM_PAINT indicates the window needs repainting in some way.
			; When processing this message, we use the BeginPaint and
			; EndPaint API functions to tell Windows we're painting.
			;
			; BeginPaint takes two parameters - the window handle we're
			; about to paint to and the address of a paint structure.
			;
			sc BeginPaint, .hwnd, ._ps
			;
			; It returns a handle that we can use to draw on the window with,
			; so we save it temporarily in a local variable
			;
			mov	._hdc, eax
			;
			; GetClientRect populates a rectangle structure with the limits
			; of the client area of the window we can draw on
			;
			sc GetClientRect, .hwnd, ._rect
			;
			; We then use DrawText to draw some text on the window, centering
			; it vertically and horizontally.
			;
			TEXTlocal szWndText, 'Hello, world from x86 assembler!',0
			sc DrawText, ._hdc, .szWndText, -1, ._rect, DT_SINGLELINE | DT_CENTER | DT_VCENTER
			;
			; After all our drawing, we tell Windows we've finished painting
			;
			sc EndPaint, .hwnd, ._ps
			;
			; Finally, we set a return code for the window procedure.
			; This is always zero if we have handled the message
			;
			xor	eax,eax
			;
			; And break from this case of course
			;
			break
		case WM_DESTROY
			;
			; Close button pressed, Alt-F4 etc. so tell Windows
			; we'd like to exit, the parameter passed to PostQuitMessage
			; is the return code we'd like to use, see the help for full details
			;
			sc PostQuitMessage, 0
			xor	eax,eax
			break
		default
			;
			; Any message that we don't want to handle ourselves can
			; be sent back to Windows to process in a default way:
			;
			sc DefWindowProc, .hwnd, .message, .wParam, .lParam
			;
			; Note that we don't set eax as it's already set by
			; DefWindowProc (to a non-zero value)
			;
	switchend
	;
	; Restore the registers
	;
	CallbackEpilogue
endproc

[section .data]
